<?php
namespace App\Http\Controllers;

use App\Models\Channel;
use App\Models\Post;
use App\Models\Story;
use App\Models\Topic;
use Spatie\Sitemap\Sitemap;
use Spatie\Sitemap\Tags\Url;

class SitemapController extends Controller
{
    public function index()
    {
        ob_clean();

        $sitemap = Sitemap::create();

        $now = now(); // current timestamp for static pages

        // Static pages at start
        $staticPagesStart = [
            '/'           => 1.0,
            'channels'    => 0.9,
            'topics'      => 0.9,
            'posts'       => 0.9,
            'webstories'  => 0.9,
            'videos'      => 0.9,
            'membership'  => 0.7,
            'sponsor-ads' => 0.9,
        ];

        foreach ($staticPagesStart as $path => $priority) {
            $sitemap->add(
                Url::create(url($path))
                    ->setPriority($priority)
                    ->setLastModificationDate($now)
            );
        }

        // Dynamic sections: posts, channels, topics
        $sections = [
            'posts'    => Post::where('status', 'active')->latest('publish_date')->take(15),
            'channels' => Channel::where('status', 'active')->latest()->take(10),
            'topics'   => Topic::where('status', 'active')->latest()->take(10),
        ];

        foreach ($sections as $prefix => $query) {
            foreach ($query->get() as $item) {
                $lastMod = $item->updated_at ?? $item->created_at ?? $now;
                $sitemap->add(
                    Url::create(url("$prefix/{$item->slug}"))
                        ->setPriority(0.8)
                        ->setChangeFrequency('weekly')
                        ->setLastModificationDate($lastMod)
                );
            }
        }

        // Stories
        $stories = Story::with(['story_slides', 'topic'])
            ->whereHas('story_slides')
            ->whereHas('topic')
            ->latest()
            ->take(10)
            ->get();

        foreach ($stories as $story) {
            $lastMod = $story->updated_at ?? $story->created_at ?? $now;
            $sitemap->add(
                Url::create(url("webstories/{$story->topic->slug}/{$story->slug}"))
                    ->setPriority(0.8)
                    ->setChangeFrequency('weekly')
                    ->setLastModificationDate($lastMod)
            );
        }

        // Static pages at end
        $staticPagesEnd = [
            'my-account'          => 0.5,
            'terms-and-condition' => 0.3,
            'privacy-policy'      => 0.3,
            'about-us'            => 0.6,
            'contact-us'          => 0.6,
        ];

        foreach ($staticPagesEnd as $path => $priority) {
            $sitemap->add(
                Url::create(url($path))
                    ->setPriority($priority)
                    ->setLastModificationDate($now)
            );
        }

        return response($sitemap->render(), 200)
            ->header('Content-Type', 'application/xml');
    }
}
